/*---------------------------------------------------------------------------------
Name               : device.c
Author             : Marvin Raaijmakers
Description        : Contains functions that read information from the event device
Date of last change: 12-Nov-2005

    Copyright (C) 2005 Marvin Raaijmakers

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or any later version.
    
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

    You can contact me at: marvinr(at)users(dot)sf(dot)net
    (replace (at) by @ and (dot) by .)
-----------------------------------------------------------------------------------*/
#include <linux/input.h>
#ifndef MSC_RAW
#	define MSC_RAW	0x03
#endif
#ifndef MSC_SCAN
#	define MSC_SCAN	0x04
#endif
#ifndef EV_SYN
#	define EV_SYN 0
#endif

#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>

#include <mxml/mxml.h>
#include <keytouch-editor.h>


int
get_bus_type (int event_device)
/*
Input:
	event_device	- File descriptor to an event device
Output:
	-
Returns:
	The bus type where the device is connected to.
Desciption:
	See above.
*/
{
	unsigned short id[4];
	
	ioctl (event_device, EVIOCGID, id);
	
	return ((int) id[ID_BUS]);
}


Boolean
print_device_info (int event_device)
/*
Input:
	event_device	- File descriptor to an event device
Output:
	-
Returns:
	FALSE if an error occurred, otherwise TRUE.
Desciption:
	This function reads and prints the folowing information about event_device:
	- Input driver version
	- Input device ID
	- Input device name
*/
{
	int version;
	unsigned short id[4];
	char name[256] = "Unknown";
	
	if (ioctl(event_device, EVIOCGVERSION, &version))
	{
		perror("keytouch-editor: can't get version");
		return (FALSE);
	}

	printf ("Device information:\n");
	printf (" Input driver version is %d.%d.%d\n",
		version >> 16, (version >> 8) & 0xff, version & 0xff);

	ioctl (event_device, EVIOCGID, id);
	printf (" Input device ID: bus 0x%x vendor 0x%x product 0x%x version 0x%x\n",
		id[ID_BUS], id[ID_VENDOR], id[ID_PRODUCT], id[ID_VERSION]);

	ioctl (event_device, EVIOCGNAME(sizeof(name)), name);
	printf ( " Input device name: \"%s\"\n", name);
	
	return (TRUE);
}

unsigned int
get_scancode (int event_device)
/*
Input:
	event_device	- File descriptor to an event device
Output:
	-
Returns:
	The scancode of the pressed key or 0 if enter was pressed.
Desciption:
	This function waits for a key press event on event_device and returns the
	keys scancode.
*/
{
	unsigned int		scancode;
	int			readen_size,
				i;
	Boolean			enter;
	struct input_event	ev[64]; /* Event list */
	
	enter = FALSE;
	scancode = 0;
	while (!scancode)
	{
		readen_size = read(event_device, ev, sizeof(struct input_event) * 64);

		if (readen_size < (int) sizeof(struct input_event))
		{
			perror("\nkeytouch-editor: error reading");
			exit (1);
		}
		/* Read the events and stop after a key event (if there is one in the event list). */
		for (i = 0; i < readen_size / sizeof(struct input_event); i++)
		{
			/* If the event contains a scancode */
			if (ev[i].type != EV_SYN &&
			    ev[i].type == EV_MSC &&
			    (ev[i].code == MSC_RAW || ev[i].code == MSC_SCAN))
			{
				scancode = ev[i].value;
			}
			/* If the event contains a keycode */
			else if (ev[i].type != EV_SYN)
			{
				/* If the key was released */
				if (!ev[i].value)
				{
					/* Because we only want a scancode of a key press event: */
					scancode = 0;
				}
				else if (ev[i].code == KEY_ENTER)
				{
					enter = TRUE;
				}
			}
		}
	}
	if (enter)
	{
		scancode = 0;
	}
	return (scancode);
} 
